(* ::Subsubsection:: *)
(*HoughtonCamera package*)

(* ::Text:: *)
(*This impements the Houghton Camera optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/HoughtonCamera"}]*)

BeginPackage["GeometricOptics`Packages`HoughtonCamera`", {"GeometricOptics`"}]

Options[HoughtonCamera] = {OutputType->"Report", OutputLevel->"Full", CameraType->"SD"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
	CameraType can be
					"SD" Houghton's camera with symmetric doublet - no control of astigmatism
					"DDR" Houghton's Camera with doublet with different radii - no control of astigmatism
					"SDCM" Houghton's Camera with symmetrix doublet and conic mirror				
					
*)
HoughtonCamera::usage="HoughtonCamera calculates parameters for a Houghton Camera optical system.";
HoughtonCameraInputPanel::usage = "HoughtonCameraInputPanel gives the GUI panel to work with HoughtonCamera";

$HoughtonCameraInputVariables = {"f1", "spes", "diam", "\[CapitalDelta]", "Ni", "\[Theta]"};

Begin["`Private`"]

HoughtonCamera[{f1_, spes_, diam_, capitaldelta_, Ni_, theta_}, opts___]:= HoughtonCamera[f1, spes, diam, capitaldelta, Ni, theta];

HoughtonCamera[f1_, spes_, diam_, capitaldelta_, Ni_, theta_, OptionsPattern[]] := 
Quiet@Module[{\[Delta], thick, A1, A2, r11, r12, rad, fun, gr, inputs, outputs, inPanel, outPanel, title},
 		If[ArgumentsQ["HoughtonCamera", {$HoughtonCameraInputVariables, {f1, spes, diam, capitaldelta, Ni, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 5, "PackageName" -> "HoughtonCamera", "Arguments" -> <|"f1" -> f1, "spes" -> spes, "diam" -> diam, "\[CapitalDelta]" -> capitaldelta, "Ni" -> Ni, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
		
			\[Delta] = capitaldelta/f1;
			thick = Append[spes, capitaldelta];
		
			Switch[OptionValue[CameraType],
					"SD",
					(* Aberrations coefficients of the mirror and curvature radii of the paired doublet *)
					A1 = -(1/(32 f1^3));
					A2 = 1/(8 f1^2) (2 - \[Delta]);
					r11 = -((2 A1 A2 (-1 + Ni^2))/(2 A1^2 (1 + Ni) + A2^3 (-1 + Ni) Ni));
					r12 = (2 A1 A2 (-1 + Ni^2))/(2 A1^2 (1 + Ni) - A2^3 (-1 + Ni) Ni);
					(* Corresponding aberrations *)
					rad = {r11, r12, -r11, -r12};
					TotalAberrations[{r11, r12, -r11, -r12, -2f1}, thick, {{1, Ni, 1, Ni, 1, -1}}, {0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}],
					
					"DDR",
					a1 = (4 - 3 Ni - 3 Ni^2 + 2 Ni^3)/(8 Ni);
					a2 = (-1 + Ni + Ni^2 - Ni^3)/(4 Ni);
					D1 = 1/8 - 1/(4 Ni) + Ni/8 + ((-1 + Ni^2) r3)/(8 Ni (2 - \[Delta]));
					D2 = Ni/(4 (Ni^2 - 1)) (2 - \[Delta])*r3^2;
					delta1 = (a1 + a2 + D1)^2 - 4 (a1 + a2) (D1 + a1 D2);
					delta11 = Numerator[Simplify[delta1]];
					r3adm = 0.99 First@TakeLargest[Replace[r3, Solve[delta11 == 0, r3]], 1];
					(* Second solution *)
					r11 = ReplaceAll[-f1 r3 (2 a1 (a1 + a2))/(a1 (a2 - 2 D1) + a2 (a2 - D1 + Sqrt[delta1])), r3 -> (0.98 r3adm)];
					r12 = ReplaceAll[f1 r3 (2 (a1 + a2))/(a1 + a2 + D1 + Sqrt[delta1]), r3 -> (0.98 r3adm)];
					r13 = 0.98 f1 r3adm;
					r14 = (r11 r12 r13)/ (r11 (r12 - r13) + r12 r13);
					rad = {r11, r12, r13, r14};
					TotalAberrations[{r11, r12, r13, r14, -2f1}, thick, {{1, Ni, 1, Ni, 1, -1, 1}}, {0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}],
					
					"SDCM",
					\[Delta] = capitaldelta/f1;
					K = N[-((-2 + \[Delta])^2/\[Delta]^2)];
					r11 = (4 (-1 + Ni^2) (-2 + \[Delta]) (-1 + \[Delta]) \[Delta])/(-2 (-1 + \[Delta])^2 + Ni^2 (-2 + \[Delta])^3 \[Delta] - Ni (2 - 12 \[Delta] + 14 \[Delta]^2 - 6 \[Delta]^3 + \[Delta]^4)) f1;
					r12 = (4 (-1 + Ni^2) (-2 + \[Delta]) (-1 + \[Delta]) \[Delta])/(2 (-1 + \[Delta])^2 + Ni^2 (-2 + \[Delta])^3 \[Delta] + Ni (2 + 4 \[Delta] - 10 \[Delta]^2 + 6 \[Delta]^3 - \[Delta]^4)) f1;
					rad = {r11, r12, -r11, -r12};
					TotalAberrations[{r11, r12, -r11, -r12, -2 f1}, thick, {{1, Ni, 1, Ni, 1, -1}}, {0, 0, 0, 0, K}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}]
				 ];
			(* Drawing *)
			fun = MapThread[If[#1 > 0, #1 + #2 - Sqrt[#1^2 - x^2], #1 + #2 + Sqrt[#1^2 - x^2]] &, {rad, Prepend[Accumulate[spes], 0]}];
			gr = Plot[Evaluate@fun, {x, -diam/2, diam/2}, Filling -> {1 -> {2}, 3 -> {4}}, PlotRange -> All, AspectRatio->Automatic];
			  	  
			outputs = {{"Primary radius", -2 f1},
					   {"Distance between the primary and the corrector", \[Delta] f1},
					   {"The radii of Houghton's doublet are",""},
					   {"r1", r11},
					   {"r2", r12},
					   {"r3", -r11},
					   {"r4", -r12},
					   {"Height of the image", GOimageHeight[[1, GOn]]},
					   {"Focal length", GOfocalLength[[1]]},
					   {"Total spherical aberration", GOaberration[[1]]},
					   {"Total sagittal coma", GOcoma[[1]]},
					   {"Total astigmatism", GOastigmatism[[1]]},
					   {"Plots", gr}};
			
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"List of thickness of the doublet", "spes", spes},
							{"Diameter of the corrector", "diam", diam},
							{"Distance between the corrector and the mirror", "\[CapitalDelta]", capitaldelta},
							{"Refractive index of corrector", "Ni", Ni},
							{"Field angle  in degree", "\[Theta]", theta},
							{"Camera Type","", OptionValue[CameraType]}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];

			cameratype = Switch[OptionValue[CameraType], 
								"SD",
								"Symmetrix doublet - no control of astigmatism",
								"DDR",
								"Camera with doublet with different radii - no control of astigmatism",
								"SDCM",
								"Camera with symmetrix doublet and conic mirror"];
			title = BoxData[ToBoxes[Column[{$GeometricOpticsPackagesList[SelectFirst[#PackageName == "HoughtonCamera" &], "Description"], Style[cameratype, "Text"]}]]];
			
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> title,																	
												"date" -> DateString[], 
												"function" -> "HoughtonCamera", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> title, 
													"date" -> DateString[], 
													"function" -> "HoughtonCamera", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1", f1},
														 {"spes", spes},
														 {"diam", diam},
														 {"capitaldelta", capitaldelta},
														 {"Ni", Ni},
														 {"theta", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["HoughtonCamera not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

HoughtonCameraInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "HoughtonCamera", cameratype="SD"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, spes, diam, capitaldelta, Ni, theta} = ReplaceAll[$HoughtonCameraInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run HoughtonCamera function", "Text"],
					Row[{
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the focal length of the primary mirror"]},
						  {"List of thickness of the doublet", "spes", Tooltip[InputField[Dynamic[spes], Alignment -> Center], "Insert the list of thickness of the doublet"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Distance between the corrector and the mirror", "capitaldelta", Tooltip[InputField[Dynamic[capitaldelta], Alignment -> Center], "Input the distance between the corrector and the mirror"]},
						  {"Refractive index of the corrector", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Input the refractive index of the corrector"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
						
						Spacer[10],
						Column[{"Camera type", PopupMenu[Dynamic[cameratype], {"SD", "DDR", "SDCM"}]}]}],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, spes, diam, capitaldelta, Ni, theta}], OutputType -> outputtype, CameraType->cameratype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, spes, diam, capitaldelta, Ni, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]